
*-----------------------------------------------------------
* Program    :	3 Primitive Threaded Interpretor
* Written by :	Jim Gilbert
* Date       :	Version r2t - June 20, 2011
* Description:	Primitive, extensible threaded interpretor 
*		specific to 68008 but w/ portable language
*-----------------------------------------------------------
*
DBLBYTS	equ	8
DBLWRDS	equ	4
DBLLNGS	equ	2
LNGNIBS	equ	8
LNGBYTS	equ	4
LNGWRDS	equ	2
NIBBITS	equ	4
NIBMSK	equ	$0F
WRDBYTS	equ	2
WRDNIBS	equ	4
*
NULLPTR equ	0
FALSE	equ	0		* Booleans always evaluated by testing for zero/nonzero
TRUE	equ	-1 		* Using this value for TRUE means NOT TRUE == FALSE
ARGERR	equ	-1
*
*-----------------------------------------------------------
*	Mark Stack Frame Structure
	OFFSET	0
Result	DS.L	1		* a6-> result of this fn call
PrvFrm	DS.L	1		* old a6 at time of call to this fn
MARKBYTS equ	*		* Number of bytes in halfmark 
*	[DS.L	1		* arg 0]
*	..
*	[DS.L	1		* arg n]
*RetPC	DS.L	1		* Return PC
*	[DS.L	1		* local 0]
*	..
*	[DS.L	1		* local m]
	ORG	*
*-----------------------------------------------------------
*
BUFBYTS equ	80
BUFMAXIX equ 	BUFBYTS-1
BUFWDS  equ	BUFBYTS/WRDBYTS
BUFADRCHs equ	6+2		* Line width displays start with 68K addr space (24 bits)
				* worth of addr in hex which takes 6 bytes plus : plus space
BUFCHLS	equ	(BUFBYTS-BUFADRCHS)/(LNGNIBS+1)
*
*-----------------------------------------------------------
*
*
halfmark macro			* Builds the mark but doesn't update a6 yet
	clr.l	-(a7)		* Allocate and clear result
	move.l	a6,-(a7)	* Save Current Frame Pointer
	endm	

mark	macro			* Builds mark and updates a6
	halfmark
	lea.l	LNGBYTS(a7),a6	* New Frame Point -> Result
	endm
*
unmark	macro			* Cuts back the stack and restores previous a6
	move.l	a6,a7		* Cutback sp->result
	move.l	-LNGBYTS(a7),a6	* Restore Prev Frame Pointer
	endm
*
unmarkdrop macro
	unmark
	lea.l	LNGBYTS(a7),a7
	endm
*-----------------------------------------------------------
*
*	Register Assignments
*	SP	a7	Stack Ptr SET AT ENTRY; WD ALIGNED
*	GlblFP	d7	Global Frame Pointer
*	FP	a6	Frame Ptr
*	SymStr	d6	StmPtr Points to SymBuf or SymCode Vec, loworder bit overloaded
*	 IsBuf	d6:0=1 if true, d6:0=0 means IsPrt
*	SymPC	a5	CharInx into SymStr of last symcode byte fetched
*	InCode	d5	zero means NOT in bincodefn; nonzero=in bincdfn(- assemble,+interpret)
*	Avail	a4	long aligned next avail to alloc, end of pool SET AT ENTRY
*	DicLst	a3	dicptr of last dict entry  long aligned (3 longs each)
*	VecLst	a2	vecptr of last vec         long aligned (even # longs >4 in len)
*	FREE	a1-a0,d4-d0 
*	A0 =    START at ENTRY
*-----------------------------------------------------------
*
*	Global Frame - Pointed To by D7
*
	OFFSET	0
_GL_Result	DS.L	1
_GL_PrvFrm	DS.L	1
_GL_AvailInit	DS.L	1
_GL_StartPC	DS.L	1
_GL_SymStr	DS.L	1
_GL_SymPC	DS.L	1
_GL_Incode	DS.L	1
_GL_DicLstInit	DS.L	1
_GL_VecLstInit	DS.L	1
_GL_Fid		DS.L	1
_GL_SymBuf	DS.L	1
_GL_NmBuf	DS.L	1
_GL_FilBuf	DS.L	1
	ORG	*
*-----------------------------------------------------------
*
	ORG	$1000
START:				; first instruction of program
*
*	Initialize Registers and Build Global Frame; Globals are these locals	
	lea.l	-LNGBYTS(a7),a6
	move.l	a6,-(a7)		* Global Res pts to self, SP at entry - LNGBYTS
	move.l	a7,a6 			
	move.l	a7,-(a7)		* Global Prev Frame points to Global Result
	move.l	a6,d7	 		* d7 is global frame
	move.l	a4,-(a7)		* Global Avail a4 at entry
	move.l	a0,-(a7) 		* Global PC at entry
	clr.l	d6	 		* SymStr null; IsFn=false
	move.l	d6,-(a7) 		* Global SymStr: IsFn
	moveq.l	#BUFMAXIX,d5
	move.l	d5,a5	 		* SymPC = BUFMAXIX to trigger 1st strgetln
	move.l  a5,-(a7) 		* Global SymPC 
	clr.l	d5	 		* Incode=false, non-zero -> 0th cd wd on long stk
	move.l  d5,-(a7) 		* Global Incode
	lea.l	__FilGetLn,a3 		* Initial DicLst=last predefined
	move.l  a3,-(a7) 		* Global DicLstInit
	lea.l	_Fid,a2	 		* Initial VecLst=last predefined
	move.l  a2,-(a7) 		* Global VecLstInit
	move.l	a2,-(a7)		* Global _Fid
	clr.l	-(a7)			* SymBuf allocated
	clr.l	-(a7)			* NmBuf allocated
	clr.l	-(a7)			* FilBuf allocated
*	
	jmp	Initialize
*-----------------------------------------------------------
*
*	Predefined Constants, Types, and Routines for Char, Prt, Vec, Str and Dic
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Char.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Prt.x68'
	Include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Vec.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Str.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Dic.x68'
*
*	Predefined Routines for Reg, Stk, Rel, Mov and Fil
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Reg.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Stk.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Rel.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Mov.x68'
	include 'C:\Documents and Settings\Jim Gilbert\My Documents\68008\3\Fil.x68'
*-----------------------------------------------------------
*
Initialize:
*
*	Initialize SymBuf for Getln input
*	GL_SymStr=(GL_SymBuf=VecNew(0,a2,VEC_HDR_WDS+BUFWDS,VEC_ELSE_BYTE,BUFMAXIX))
	mark
	  clr.l		d0
	  move.l	d0,-(a7)	* datap
	  move.l	a2,-(a7)	* veclst
	  moveq.l	#VEC_HDR_WDS+BUFWDS,d0
	  move.l	d0,-(a7)        *dcwdsz
	  moveq.l	#VEC_ELSZ_BYT,d0
	  move.l	d0,-(a7)        * elsz
	  move.l	#BUFMAXIX,d0
	  move.l	d0,-(a7)	* curinx
	  lea.l		VecNew,a0
	  jsr		(a0)
	unmark	
*		
*	Initialize SymStr to Point to SymBuf
*	SymStr gets assigned to SymBuf, FilBuf or Sym Fn Vector as we go
	move.l	(a7)+,d6		* SymBuf vecptr
	move.l	d7,a0
	move.l	d6,-_GL_SymBuf(a0)	* Global SymBuf
	move.l	d6,-_GL_SymStr(a0)	* Global SymStr
	move.l	d6,(a6)			* enable ^ to work for user
*
*	Initialize Nmbuf for name scanning and lookup
*	tos=VecNew(0,a2,VEC_HDR_WDS+BUFWDS,VEC_ELSE_BYTE,BUFMAXIX)
	mark
	  clr.l		d0
	  move.l	d0,-(a7)	* datap
	  move.l	a2,-(a7)	* veclst
	  moveq.l	#VEC_HDR_WDS+BUFWDS,d0
	  move.l	d0,-(a7)        * dcwdsz
	  moveq.l	#VEC_ELSZ_BYT,d0
	  move.l	d0,-(a7)        * elsz
	  move.l	#BUFMAXIX,d0
	  move.l	d0,-(a7)	* curinx
	  lea.l		VecNew,a0
	  jsr		(a0)
	unmark	
*		
*	Initialize Global NmBuf to Point to Buf
	move.l	(a7)+,d0		* NmBuf vecptr
	move.l	d7,a0
	move.l	d0,-_GL_Nmbuf(a0)	* Global Symbuf
*
*	Initialize Filbuf for include file input
*	tos=VecNew(0,a2,VEC_HDR_WDS+BUFWDS,VEC_ELSE_BYTE,BUFMAXIX)
	mark
	  clr.l		d0
	  move.l	d0,-(a7)	* datap
	  move.l	a2,-(a7)	* veclst
	  move.l	#VEC_HDR_WDS+FILBUFWDS,d0
	  move.l	d0,-(a7)        * dcwdsz
	  moveq.l	#VEC_ELSZ_BYT,d0
	  move.l	d0,-(a7)        * elsz
	  move.l	#FILBUFMAXIX,d0
	  move.l	d0,-(a7)	* curinx
	  lea.l		VecNew,a0
	  jsr		(a0)
	unmark			
*	Initialize Global FilBuf to Point to FilBuf
	move.l		(a7)+,d0	* FilBuf vecptr
	move.l		d7,a0
	move.l		d0,-_GL_Filbuf(a0) * Global Symbuf
*
*	Allocate _Fid scalar
	mark
	  move.l	a3,-(a7)	* DicLst
	  pea.l		_Fid		* namptr
	  move.l	#DICTYPSCALAR,-(a7) * dictyp=scalar
	  clr.l		-(a7)		* initial value zero
	  lea.l		DicNew,a0
	  jsr		(a0)
	unmarkdrop
*
*	End Initialization	
*-----------------------------------------------------------
*
Interp:
*	Main Interpretor Loop - Done with all ChLits,HexLits,Scalars,and Names/Namecalls so far
*	but maybe in middle of assembling a BinCode Function Vector (d5=Incode non-zero)
*
*	Need to get next character here
	mark
	  lea.l	StrNxtCh,a0
	  jsr	(a0)
	unmark
*				* Already have a character
WithCh:				* tos is ch
	move.l	(a7)+,d0
	cmpi.l	#Char_SPACE,d0
	ble	Interp		* Ignore space, all ctl chars interp or assemble 
	cmpi.l	#Char_POUND,d0
	beq	startscalar	* # starts scalar literal in interp or assemble
	cmpi.l	#Char_RQUOTE,d0
	beq	startchlit	* ' starts byte vector char lit interp or assemble 
	cmpi.l	#Char_DEL,d0
	beq	Interp		* Ignore DEL chars interp or assemble
	cmpi.l	#'0',d0
	blt	trydquote
		cmpi.l	#'9',d0
		ble	declit	* 0-9 starts decimal literal
*
*	=======================  " Return Operators =====================================
trydquote:
*				* " returns tos, "? conditionally returns tos
*				* "d returns dbl tos, "?d conditionally dbl returns tos
	cmpi.l	#Char_DBLQUOTE,d0 * " does a return or assembles a return
	bne	trytilde
*		d0 = StrNxtCh()
		mark
			lea.l	StrNxtCh,a0
			jsr	(a0)
		unmark
		move.l	(a7)+,d0
*
*		if d0 == QMARK then
		cmpi.l	#Char_QMARK,d0
		bne	dqtilde
*		  d0 = StrNxtCh()			* here with "?...
		  mark
			lea.l	StrNxtCh,a0
			jsr	(a0)
		  unmark
		  move.l	(a7)+,d0
*	========================= "?d Return Double If TRUE ===================================
*		  if d0 == 'd' then			* here with "?d
		  cmpi.l	#'d',d0
		  bne   	dqqmk
*			if incode then
			tst.l	d5
			beq	dqqmkdinterp		
				move.l	#$4A9F,-(a7)	* tst.l (a7)+
				move.l	#$6700,-(a7)	* beq *+$A
				move.l	#$000A,-(a7)	* $A
	  			move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
				move.l	#$2D5F,-(a7)	* move.l (a7)+,4(a6)
				move.l	#$0004,-(a7)	* displ 4
				move.l	#$4E75,-(a7)	* rts
	  			move.l	#$4FEF,-(a7)	* lea.l 8(a7),a7
	  			move.l	#$0008,-(a7)	* displ 8
			bra	Interp
*			else
dqqmkdinterp:
				tst.l	(a7)+		* bool?
				beq	dqqmkdfalse	
				move.l	(a7)+,(a6)	* result = tos+
				move.l	(a7)+,LNGBYTS(a6) * loworder result = tos+
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp
dqqmkdfalse:
				lea.l	2*LNGBYTS(a7),a7 * drop return value
			bra	Interp
*	  	endif
dqtilde:
*		if d0 == '~' then
		cmpi.l		#Char_TILDE,d0
		bne		dqtryd
*		  d0 = StrNxtCh()			* here with "~...
		  mark
			lea.l	StrNxtCh,a0
			jsr	(a0)
		  unmark
		  move.l	(a7)+,d0
*	========================== "~d Return Double if FALSE ============================
*		  if d0 == 'd' then			* here with "~d
		  cmpi.l	#'d',d0
		  bne   	dqtildex
*			if incode then
			tst.l	d5
			beq	dqtildedinterp		
				move.l	#$4A9F,-(a7)	* tst.l (a7)+
				move.l	#$660A,-(a7)	* beq *+$A
	  			move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
				move.l	#$2D5F,-(a7)	* move.l (a7)+,4(a6)
				move.l	#$0004,-(a7)	* displ 4
				move.l	#$4E75,-(a7)	* rts
	  			move.l	#$4FEF,-(a7)	* lea.l 8(a7),a7
	  			move.l	#$0008,-(a7)	* displ 8
			bra	Interp
*			else
dqtildedinterp:
				tst.l	(a7)+		* bool?
				beq	dqtildedfalse	
				move.l	(a7)+,(a6)	* result = tos+
				move.l	(a7)+,LNGBYTS(a6) * loworder result = tos+
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp
dqtildedfalse:
				lea.l	2*LNGBYTS(a7),a7 * drop return value
			bra	Interp
dqtildex:
*	========================= "~ Return if FALSE ==============================
*			if incode then		
			tst.l	d5
			beq	dqtildeinterp
				move.l	d0,-(a7)	* put the non-d character back
				move.l	#$4A9F,-(a7)	* push a tst.l (a7)+ into code on stk
				move.l	#$6606,-(a7)	* push bne *+6 into code on stk
				move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
	  			move.l	#$4e75,-(a7)	* rts instruction to code on stk
	  			move.l	#$4FEF,-(a7)	* lea.l	4(a7),a7
	  			move.l	#$0004,-(a7)	* displ 4
			bra	WithCh
*			else
dqtildeinterp:
				tst.l	(a7)+		* bool?
				bne	dqtildetrue	
				move.l	(a7)+,(a6)	* TRUE; exit cycle-call result = tos+
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp		* next char comes from returned to locn, often a }
dqtildetrue:
				lea.l	LNGBYTS(a7),a7  * drop return value stay in callee
			bra	Interp
*		endif
*
*	======================== "? Return if TRUE ===============================
dqqmk:							* here with "? 
*			if incode then		
			tst.l	d5
			beq	dqqmkinterp
				move.l	d0,-(a7)	* put the non-d character back
				move.l	#$4A9F,-(a7)	* push a tst.l (a7)+ into code on stk
				move.l	#$6706,-(a7)	* push beq *+6 into code on stk
				move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
	  			move.l	#$4e75,-(a7)	* rts instruction to code on stk
	  			move.l	#$4FEF,-(a7)	* lea.l	4(a7),a7
	  			move.l	#$0004,-(a7)	* displ 4
			bra	WithCh
*			else
dqqmkinterp:
				tst.l	(a7)+		* bool?
				beq	dqqmkfalse	
				move.l	(a7)+,(a6)	* TRUE; exit cycle-call result = tos+
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp		* next char comes from returned to locn, often a }
dqqmkfalse:
				lea.l	LNGBYTS(a7),a7  * drop return value stay in callee
			bra	Interp			
		
*
*	======================= "d Unconditional Return Double ================
dqtryd:	      * have " see if it is "d
*	      if d0 == d then
	      cmpi.l	#'d',d0
	      bne	dqalone
*							* here with "d
*			if incode then
			tst.l	d5
			beq	dqdinterp		
				move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
				move.l	#$2D5F,-(a7)	* move.l (a7)+,4(a6)
				move.l	#$0004,-(a7)	* displ 4
	  			move.l	#$4e75,-(a7)	* rts
			bra	Interp			* don't have next char yet
*			else
dqdinterp:
				move.l	(a7)+,(a6)	* result = tos+
				move.l	(a7)+,LNGBYTS(a6) * loworder result for double
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp			* next char will come from place returned to
*	      else
*
*	========================== " Unconditional Return =======================	
dqalone:						* " sets result to tos then returns
*			if incode then
			tst.l	d5
			beq	dqaloneinterp		
				move.l	#$2C9F,-(a7)	* move.l (a7)+,(a6)
	  			move.l	#$4E75,-(a7)	* rts
			bra	WithCh			* use next char for compiled unconditional return
*			else
dqaloneinterp:
				move.l	(a7)+,(a6)	* result = tos+
				move.l	(a7)+,a5
				move.l	(a7)+,d6
			bra	Interp			* next char will come from place returned to
*	
*	=======================  TILDE ========================================
trytilde:
	cmpi.l	#Char_TILDE,d0		* ~ means bitwise NOT
	bne	tryvbar
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l (a7)+,d0
	  cmpi.l #Char_TILDE,d0
	  bne	interptilde		* ~~ is compiled if incode else same as ~
	  	tst.l	d5
	  	beq	interptilde
	  		move.l	#$201F,-(a7)	* move.l(a7)+,d0
			move.l	#$4680,-(a7)	* not.l	d0
			move.l	#$2F00,-(a7)	* move.l d0,-(a7)	
	bra	Interp
interptilde:
	  move.l (a7)+,d1
	  not.l	 d1
	  move.l d1,-(a7)
	  move.l d0,-(a7)
	bra   WithCh
*
*	=======================  VBAR ========================================
tryvbar:
	cmpi.l	#Char_VBAR,d0	* | is bitwise OR
	bne	trybang
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l 	(a7)+,d0
	  cmpi.l	#Char_VBAR,d0
	  bne		interpvbar	* || is compiled if incode else same as |
	  	tst.l	d5
	  	beq	interpvbar
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$8081,-(a7)	* or.l	 d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	bra	Interp
interpvbar:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  or.l		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  BANG ========================================
trybang:
	cmpi.l	#Char_BANG,d0	* ! is exclusive or at interp time
	bne	tryand	
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_BANG,d0
	  bne		interpbang	* !! is compiled if incode else same as !
	  	tst.l	d5
	  	beq	interpbang
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$B380,-(a7)	* eor.l	 d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)  
	  bra	Interp	
interpbang:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  eor.l		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  AMPER ========================================
tryand:	
	cmpi.l	#Char_AMPER,d0 * & is bitwise AND
	bne	tryplus
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_AMPER,d0
	  bne		interpand	* && is compiled if incode else same as &
	  	tst.l	d5
	  	beq	interpand
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$C081,-(a7)	* and.l	 d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)  	
	bra	Interp
interpand:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  and.l		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  PLUS ========================================
tryplus:
	cmpi.l	#Char_PLUS,d0	* + is plus
	bne	tryminus
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_PLUS,d0
	  bne		interpplus	* ++ is compiled if incode else same as +
	  	tst.l	d5
	  	beq	interpplus
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		add.l	#$D081,-(a7)	* add.l	 d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	  bra	Interp
interpplus:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  add.l		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  MINUS ========================================
tryminus:
	cmpi.l	#Char_MINUS,d0	* - is minus
	bne	trystar
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_MINUS,d0
	  bne		interpminus	* -- is compiled if incode else same as -
	  	tst.l	d5
	  	beq	interpminus
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		add.l	#$9081,-(a7)	* sub.l	 d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	bra	Interp
interpminus:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  sub.l		d2,d1
	  move.l	d1,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  STAR ========================================
trystar:
	cmpi.l	#Char_STAR,d0	* * is unsigned 16 bit multiply
	bne	tryslash
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_STAR,d0
	  bne		interpstar	* ** is compiled if incode else same as *
	  	tst.l	d5
	  	beq	interpstar
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$C0C1,-(a7)	* mulu.l d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	  bra	Interp
interpstar:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  mulu		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  SLASH ========================================
tryslash:
	cmpi.l	#Char_SLASH,d0	* / is unsigned 16 bit divide
	bne	trylshift	* leaves highorder wd remainder
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_SLASH,d0
	  bne		interpslash	* // is compiled if incode else same as /
	  	tst.l	d5
	  	beq	interpslash
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$80C1,-(a7)	* divu.l d1,d0
	  		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	bra	Interp
interpslash:
	  move.l	(a7)+,d2 * loworder word quotient
	  move.l	(a7)+,d1
	  divu		d1,d2
	  move.l	d2,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  LPAREN ========================================
trylshift:
	cmpi.l	#Char_LPAREN,d0	* ( is left shift logical
	bne	tryrshift
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_LPAREN,d0
	  bne		interplshift	* (( is compiled if incode else same as (
	  	tst.l	d5
	  	beq	interplshift
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$E3A9,-(a7)	* lsl.l d0,d1
	  		move.l	#$2F01,-(a7)	* move.l d1,-(a7)
	bra	Interp
interplshift:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  lsl.l		d2,d1
	  move.l	d1,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  RPAREN ========================================
tryrshift:
	cmpi.l	#Char_RPAREN,d0	* ) is right shift logical
	bne	trylbrace
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
	  cmpi.l	#Char_RPAREN,d0
	  bne		interprshift	* )) is compiled if incode else same as )
	  	tst.l	d5
	  	beq	interprshift
	  		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	  		move.l	#$221F,-(a7)	* move.l (a7)+,d1
	  		move.l	#$E0A9,-(a7)	* lsl.l d0,d1
	  		move.l	#$2F01,-(a7)	* move.l d1,-(a7)
	  bra	Interp
interprshift:
	  move.l	(a7)+,d2
	  move.l	(a7)+,d1
	  lsr.l		d2,d1
	  move.l	d1,-(a7)
	  move.l	d0,-(a7)
	bra	WithCh
*
*	=======================  LBRACE Operators =============================
trylbrace:			* Assemble a mark or do one
	cmpi.l	#Char_LBRACE,d0	* { is mark
	bne	tryrbrace
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
*	======================= {: Undermark ===================================
	  cmpi.l	#Char_COLON,d0 	* {: is undermark
	  bne	trydbllbrace   * pop tos, tos-1, mark, repush
		tst.l	d5
		beq	interpundermark
			move.l	#$201F,-(a7)	* move.l (a7)+,d0
			move.l	#$221F,-(a7)	* move.l (a7)+,d1
			move.l	#$42A7,-(a7)	* clr.l	-(a7)
			move.l	#$2F0E,-(a7)	* move.l a6,-(a7)
			move.l	#$4DEF,-(a7)	* lea.l	LNGBYTS(a7),a6
			move.l	#$0004,-(a7)	* displ LNGBYTS
			move.l	#$2F01,-(a7)	* move.l d1,-(a7)
			move.l	#$2F00,-(a7)	* move.l d0,-(a7)
	bra 	Interp
interpundermark:
	  		move.l	(a7)+,d0
	  		move.l	(a7)+,d1
	  		mark
	  		move.l	d1,-(a7)
	  		move.l	d0,-(a7)
	bra	Interp
*
*	====================== {{ Mark Double ===================================
trydbllbrace:
	  cmpi.l	#Char_LBRACE,d0	
	  bne	interplrbrace		
		tst.l	d5
		beq	interpdbllbrace
			move.l	#$42A7,-(a7)	* clr.l	-(a7)
			move.l	#$2F0E,-(a7)	* move.l a6,-(a7)
			move.l	#$4DEF,-(a7)	* lea.l	LNGBYTS(a7),a6
			move.l	#$0004,-(a7)	* displ LNGBYTS
	bra 	Interp
interpdbllbrace:
	  	clr.l	-(a7)	* Non-incode {{ pushes room for a dbl fn result
	  	mark
	bra	Interp
*
*	===================== {} Mark Call Unmark Until True ===================
interplrbrace:
	  cmpi.l	#Char_RBRACE,d0		* {} is cycle-invoke of tos dicent until true
	  bne		interplrtilde
	  	move.l	(a7)+,d0		* dicent SymPCold SymStrold oldpars
	  	move.l	(a7)+,d1
	  	move.l	(a7)+,d2		
	  	move.l	d0,-(a7)		
	  	move.l	d2,-(a7)
	  	move.l	d1,-(a7)		* SymPCold SymStrold dicent old pars
	  	mark				* mark SymPCold SymStrold dicent oldpars
	  	move.l	d0,-(a7)		* dicent mark SymPCold SymStrold dicent oldpars
	  	move.l	#1,d0
		sub.l	d0,a5			* backup SymPC by 1 so fetch on return gets } of {}
	  	move.l	#Char_LBRACE,-(a7)	* invoke expects a char so it will have {
  	  bra	Invoke			     * will be: SymPC SymStr { dicent mark  SymPCold SymStr dicent old	
*
*	===================== dicent@ {~} Mark Call Unmark Until False ===========
interplrtilde:	
	  cmpi.l	#Char_TILDE,d0		* {~} is cycle-invoke of tos dicent@ until false
	  bne		tryforloop		* end up here each time around due to } backing up
	  	move.l	(a7)+,d0		* dicent SymPCold SymStrold oldpars
	  	move.l	(a7)+,d1
	  	move.l	(a7)+,d2		
	  	move.l	d0,-(a7)		
	  	move.l	d2,-(a7)
	  	move.l	d1,-(a7)		* SymPCold SymStrold dicent old pars
	  	mark				* mark SymPCold SymStrold dicent oldpars
	  	move.l	d0,-(a7)		* dicent mark SymPCold SymStrold dicent oldpars
	  	move.l	#Char_TILDE,-(a7)	* invoke expects a char so it will have ~
	  bra	Invoke				* fetch on return will get } of {~}
*	
*	===================== dicent@ i {%} Mark Call UnMark Until i < 0 ========
tryforloop:
	  cmpi.l	#Char_PCT,d0		* cyclic re-invoke dicent@ i {%} until stk copy i < 0
	  bne		interplbrace		* end up here each time around due to } backing up
	  	move.l	(a7)+,d0		* i dicent@ oldpars
	  	move.l	(a7)+,d1		* dicent@ oldpars
	  	move.l	d0,-(a7)		* i olpars
	  	move.l	d1,-(a7)		* dicent@ i oldpars
	  	mark				* mark  dicent@ i oldpars
	  	move.l	d0,-(a7)		* i mark dicent@ i oldpars
	  	move.l	d1,-(a7)		* dicent@ i mark dicent@ i oldpars
	  	move.l	#Char_PCT,-(a7)		* invoke expects a char so it will have %
	  bra		Invoke			* invoke drops % dicent@, i is arg to invoked fn
*						* fetch on return will get } of {%}
*
*	===================== { Unmark ==================================== 
interplbrace:
	  	mark
	  	move.l	d0,-(a7) 		* Put char that isn't an LBRACE back
	bra	WithCh
*
*	=======================  RBRACE Operators ========================================
tryrbrace:			* Assemble an unmark or do one

	cmpi.l	#Char_RBRACE,d0		* } is unmark
	bne	trybkslash
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
*	======================= }} Unmark Drop ===========================================
	  cmpi.l	#Char_RBRACE,d0
	  bne	interprbrace
		tst.l	d5
		beq	interpdblrbrace		* }} incode compiles to stk
			move.l	#$2E4E,-(a7)	* move.l a6,a7
			move.l	#$2C6F,-(a7)	* move.l -4(a7),a6
			move.l	#$FFFC,-(a7)	* displ -4
	  bra Interp
interpdblrbrace:		* Non-incode }} is unmarkdrop
		  unmark
		  lea.l	LNGBYTS(a7),a7		* drop
	  bra	Interp
*
*	===================== } Unmark alone or just after {in {} or ~ in {~} or {% in {%} ==========
interprbrace:
	  	unmark				* } interp is unmark
	  	move.l	d0,d3			* have scanned char after } which isn't } so save it incase
*	 	d0 = VecEltI(SymStr,a5-2,true)	* no backup below
		mark
    			moveq.l #StrBufISFN,d0  * d1=ISFN mask
     			not.l   d0		* d0=SymBuf Addr mask 
     			and.l   d6,d0		* d0=a0=d6 & d0
     			move.l	d0,-(a7)	* clean SymStr
			move.l	a5,d0
			subq.l	#2,d0
			move.l	d0,-(a7)
			moveq.l #1,d0
			move.l  d0,-(a7)
			lea.l	VecEltI,a0
			jsr	(a0)
		unmark
		move.l	(a7)+,d0		* d0 is char immediately preceding }
*
*	===================== { of {} If TRUE backup and recall else Unmark =============
*	        if d0 == LBRACE then		* {} cycle-invoke of prior dicent@ detected
		cmpi.l	#Char_LBRACE,d0		* 
		bne	interprbtrytilde
*		  if tos == 0 then
		  tst.l	(a7)			* if fn result false we'll loop
		  bne	interprbdropdicent	* else we're done looping and will continue
interprbloop:					* looping code for {} and {~} condition not met
*			SymPC -= 3		* scanner was ready to fetch 1st after }
			move.l	#3,d0		* now ready to fetch { again
rbracebackup:
			sub.l   d0,a5		* back up interp; next StrNxtCh rets { of {} {~} to loop
*			* Rotate dicent up from below SymPC and Symstr
			lea.l	LNGBYTS(a7),a7	* drop bool
			move.l	(a7)+,d0	* pop SymPC
			move.l	(a7)+,d1	* pop SymStr
			move.l	(a7)+,d3	* pop dicent@
			move.l	d1,-(a7)	* repush SymStr
			move.l	d0,-(a7)	* repush SymPC
			move.l	d3,-(a7)	* repush dicent@
*		  	goto  Interp		* dicent@ SymPC SymStr
		  bra	Interp
*	        endif
interprbdropdicent:				* bool result we wanted, remove the saved dicent from below
		move.l	(a7)+,d4		* pop result
		move.l	(a7)+,d0		* pop SymPCOld
		move.l	(a7)+,d1		* pop SymStrOld
		lea.l	LNGBYTS(a7),a7		* drop dicent@
		move.l	d1,-(a7)		* repush SymStrOld
		move.l	d0,-(a7)		* repush SymPCOld
		move.l	d4,-(a7)		* repush result
interprbnobackup:
	  	move.l	d3,-(a7)		* Put char that isn't an RBRACE back
	bra	WithCh				* resume normal interpreting without backup
*
*	=================== ~ of dicent@ {~} if FALSE backup and recall else unmark ================
interprbtrytilde:
*		if d0 == TILDE then		* {~} cycle invoke of prior name@ detected
		cmpi.l	#CHAR_TILDE,d0
		bne	interprbtrypct
*		  if tos <> zero then
		  tst.l	(a7)
		  beq	interprbdropdicent	* if fn result false we're done
rbracebackup4:
*			SymPC -= 4
			move.l	#4,d0
*		        goto 	rbracebackup
		  	bra	rbracebackup
*		  endif
*	        endif
*
*	=================== } of dicent@ i {%}  if i-- >= 0 then backup and recall else unmark ===
interprbtrypct:					* result dicent@ i old
*		if d0 == PCT then
		cmpi.l	#CHAR_PCT,d0
		bne	interprbnobackup
*		  d0 = i
		  move.l	2*LNGBYTS(a7),d0
*		  i = d0 - 1
		  subq.l	#1,d0
		  move.l	d0,2*LNGBYTS(a7) * result dicent@ i old
*		  if d0 < 0 then 		
		  bge	interprbpctnext 	* just did zeroth case, we're done
*						* remove saved dicent and i from under
			move.l	(a7)+,d4	* pop result
			lea.l	2*LNGBYTS(a7),a7 * drop dicent@ and i
			move.l	d4,-(a7)	* repush result
		  bra	interprbnobackup	
*		  else
interprbpctnext:				* i >= 0 so more loops to make
*		  * Rotate dicent and i up from below SymPC and Symstr
		  	lea.l	LNGBYTS(a7),a7	* drop result: dicent@ i old
			move.l	(a7)+,d2	* dicent@
			move.l	(a7)+,d4	* i
			move.l	d2,-(a7)	* dicent@
			move.l	d4,-(a7)	* i
*		    	SymPC -= 4		* next StrNxtCh returns { of {%}
			move.l	#4,d0
			sub.l	d0,a5
			bra	Interp
*		  endif		
*
*	=======================  \ Comment to eol ========================================
trybkslash:			* Check for \ cmt to null/eol while in $ binfn lit
	cmpi.l	#Char_BKSLASH,d0
	bne	trydollar
flushline:			* \ anywhere except in charlit starts cmt to null/eol
	  mark			* charlit case handled there, not here
	    lea.l	StrNxtCh,a0
	    jsr	(a0)
	  unmark
	  move.l (a7)+,d0
	  cmpi.l #Char_NUL,d0
	  beq	 Interp
	  cmpi.l #Char_CR,d0
	  beq	 Interp
	bra	 flushline
*
*	=======================  $ Start/End Code Function Vector Literal Compiling ====
trydollar:
	cmpi.l	#Char_DOLLAR,d0 * $ starts and ends binary code function assembler
	bne	tryatsign
*				* Have DOLLAR sign here
	  tst.l	d5		* if not incode  
	  beq	starthxlit	* goto starthxlit
          bra	endhxlit	* else goto endhxlit
*
*	=======================  @ Namecall or @} Indexed Case Call ===============
tryatsign:
	cmpi.l	#Char_AT,d0	* @ is name call at interp time
	bne	trypct
*
	mark
		lea.l	StrNxtCh,a0
		jsr	(a0)
	unmark
	move.l	(a7)+,d0
	cmpi.l	#Char_RBRACE,d0
	beq	caseinx
	move.l	#1,d3		* back up scanner so next works 
	sub.l	d3,a5
	bra	namecall	* DicPtr to name in tos-1
*
*	======================== @} Indexed Case Call ============================
	* Usage: { fni-1@ fni-2@ ... fn0@ inxexpr i %}
caseinx:
	move.l	#1,d3
	sub.l	d3,a5		* back scanner up so next yields }
	move.l	(a7)+,d0	* i
	bgt	tryinx		* i <= 0 means no cases, just let } will cut back
badinx:
	move.l #ARGERR,(a6)
	bra	Interp
tryinx:
	move.l	(a7)+,d1	* inxexpr
	blt	badinx		* inxexpr < 0
	cmp.l	d1,d0		* i - inexpr
	ble	badinx		* inxexpr >= i
	asl	#2,d1		* inexpr * 4
	move.l	(a7,d1.l),a0	* fnj
	bra	doa0body
*
*	======================= % Invoke ==================
trypct:
	cmpi.l	#Char_PCT,d0	* % is value call i.e. an explicit invoke operator
	beq	defdname
*
*	==== ; If Incode nonzero then negate Incode so operators and predefs compile  ===
	cmpi.l	#Char_SEMI,d0	* ; pushes frameptr so can r/w result in cur frame
	bne	trycolon
*	if incode then 
	tst.l	d5
	beq	interpsemi 
		not.l	d5	* d5 negative so causes predefs to assemble,not interp 
	bra	tosjsr		* then treat it as `
*
*	=============== ; w/ Incode zero Set/Restore Breakpoint ====================
interpsemi:			* Interpreted semicolon operator reads word from (tos-1)
				* stores tos-- word at (tos-1), and returns word read
				* used for setting and restoring breakpoints
	   	move.l	LNGBYTS(a7),a0	* get target addr from tos-1
	   	clr.l	d0		* clear highorder for word load
	   	move.w	(a0),d0		* get current instruction
	   	move.l	(a7)+,d1	* pop new instruction (TRAP or code to restore)
	   	move.w	d1,(a0)		* store the replacement instruction word
	   	move.l	d0,(a7)		* return the word replaced
	bra	Interp
*
*	======================= : Colon Control Flow/Labelling Operators ============
trycolon:			* : introduces ctl flow operators or defines labels
	cmpi.l	#Char_COLON,d0  * : alone is used to define labels in codefns,symfns
	bne	trylquote
*
*		if-then:			<chlit/symfn> <bool> :]
*		unless:				<chlit/symfn> <bool> :[
*		if-then-else:	<chlit/symfn> 	<chlit/symfn> <bool> :?
*
*		Multiline nature of include file and interactive input
*		without trailing log of passed source precludes backward
*		branching (while, repeat, cycle) constructs at top level for
*		spans longer than the 1-2 chars involved in {}; must
*		implement them using : and ^ which only work within
*		current fn and/or line. There is no way to get
*		backward in the interpreted code except within the
*	        current function or buffer in which case a label works fine.
*
*	  d0=StrNxtCh()
	  mark
	  	lea.l	StrNxtCh,a0
	  	jsr	(a0)
	  unmark
	  move.l	(a7)+,d0
*
*	=========================== :] if-then =============================
*
*	  if d0 == RBRACK then
	  cmpi.l	#Char_RBRACK,d0		* :] if-then
	  bne		clntrylbrack 
* 	  	* bool,then-fn-dicent, mark from user older...
                move.l  (a7)+,d0	* then-fn-dicent, mark, older
*		if (d0 = tos+) == 0) then
		tst.l	d0		* bool?
		bne	dotosbody
dropcontinue:
			lea.l	LNGBYTS(a7),a7	* drop body on false
*			* return false
			clr.l	(a6)
*		else
		bra	Interp
doa0body:
			move.l	a0,-(a7)	* push dicent
dotosbody:
			clr.l	-(a7)		* push a NUL since invoke expects a char to discard
*			goto Invoke
			bra	Invoke
*		endif
*
*	======================= :[ unless ================================
*
*	  else if d0 == LBRACK then
clntrylbrack:
	  cmpi.l	#Char_LBRACK,d0		* :[ unless
	  bne		clntryqmark
* 	  	* bool,then-chvec-symfn-nm, users mark, older..
                move.l  (a7)+,d0
* 		if (d0 = tos+) == 0) then
	  	tst.l	d0		* bool?
*			goto dotosbody
	  		beq	dotosbody
*		else
*			goto 	dropcontinue
	  		bra	dropcontinue
*		endif
*
*	===================== :? if-then-else ==============================
*
*	  else if d0 == QMARK then
clntryqmark:
	  cmpi.l	#Char_QMARK,d0		* :? if-then-else
	  bne		dblcolon
* 	  	* bool,else-chvec-symfn-nm,then-chvec-symfn-nm,mark,older...
                move.l  (a7)+,d0 * bool
                move.l  (a7)+,a0 * else
                move.l	(a7)+,a1 * then
*		if do == 0 then
 		tst.l	d0			* bool?
*			goto	doa0body
 			beq	doa0body		* go do elsebody in a0
*		else
*			a0 = a1				* then body
 			move.l	a1,a0			* then body
*			goto 	doa0body
 			bra	doa0body		* go do then body in a0
*		endif
*	  else 
*
*	================ :: Push Addr of local n ============================
*		
dblcolon:	
	  cmpi.l	#Char_COLON,d0	  * :: repl tos by addr of tos-th local
	  bne		clnrbrace
*	  	if incode then
	 	tst.l	d5	
		beq	dblcolona	
	    		move.l	#$201F,-(a7)	* move.l (a7)+,d0
	    		move.l	#$E588,-(a7)	* lsl.l	#2,d0
	    		move.l	#$D08E,-(a7)	* add.l a6,d0
	    		move.l	#$2F00,-(a7)	* move.l d0,-(a7)
*	  	else
		bra	Interp
dblcolona:
*	    		d0 = tos+ << 2
	    		move.l	(a7)+,d0
	    		lsl.l	#2,d0
*			d0 += FRAMEP
			add.l	a6,d0
*			push	d0
			move.l	d0,-(a7)
*	    		goto	Interp
            	bra	Interp
*	  	endif
*	  endif
*	endif
*
*	========================== :} Case Branch ==============================
*
clnrbrace:
	cmpi.l		#Char_RBRACE,d0	* :} case branch operator
	bne		clnonly
* Usage: { fn1@ match1 fn2@ match2 ... fni@ matchi val i :}
		move.l	#1,d3
		sub.l	d3,a5		* back scanner up so next yields }
		move.l	(a7)+,d0	* i
		bgt	trycase		* i <= 0 means no more, } will cut back
			move.l #ARGERR,(a6)
		bra	Interp
trycase:
		move.l	(a7)+,d1	* val
		move.l	(a7)+,d2	* matchj 
		move.l	(a7)+,a0	* fnj
		cmp.l	d1,d2		* does matchj == val?
		beq	doa0body	* invoke fnj on match
*					* no match
		sub.l	d3,a5		* back scanner up so next yields :
		sub.l	d3,d0		* n = n - 1
		move.l	d1,-(a7)	* put val back
		move.l	d0,-(a7)	* put n - 1 back
		bra	Interp		* next scan will get take us back to clnrbrace
*
*	================================ : Define Label for GOTO ======================
*
clnonly:				* : followed by char not ] [ or ?
*	  	if incode then
	 	tst.l	d5	
*	    		goto 	midhxlit	
	    		bne		midhxlit	* incode colon
*	  	else
*	    		push	SYMPC
	    		move.l	a5,-(a7) 	* : in symcodefn just pushes a5=SymPC
*			push	d0
			move.l	d0,-(a7)
*	    		goto	WithCh
            		bra	WithCh
*	  	endif
*	  endif
*	endif
*
*	=======================  ` Invoke Predefined Fn Via Predef'd Dicent@ ========
trylquote:
	cmpi.l	#Char_LQUOTE,d0 * ` is JSR thru TOS at interp time
	bne	tryqmark
*		if incode then
		tst.l	d5
		beq	tosjsr
			mark
			  lea.l	StrNxtCh,a0
			  jsr	(a0)
			unmark
*			if ((d0 =(tos)) == LQUOTE) then
			move.l	(a7)+,d0
			cmpi.l	#Char_LQUOTE,d0
			bne	tosjsrputback
				move.l	#$205f,-(a7)	* move.l (a7)+,a0
				move.l	#$4E90,-(a7)	* jsr	(a0)
			bra	Interp
*			else
tosjsrputback:
*				* SymPC -= 1
				moveq.l	#-1,d0
				add.l	d0,a5
*			endif
*		endif
tosjsr:				* jsr through address in tos, 
	    move.l  (a7)+,a0    * assume mark,args and unmark all invoker's job
	    jsr	    (a0)
	bra	Interp
*	endif
*
*	=======================  ? GOTO (tos-1) if TRUE =============================
tryqmark:
	cmpi.l	#Char_QMARK,d0	* ? is jmp to code at v[tos-1] if tos true
	bne	tryperiod	* either interps it or assembles runtime code to stack
*	  d1 = StrNxtCh()
	  mark			* peek ahead to see if ?? or just ?
		lea.l	StrNxtCh,a0
		jsr	(a0)	
	  unmark
	  move.l	(a7)+,d1
*	  if d1 == QMARK then
	  cmpi.l	#Char_QMARK,d1
	  bne	ctosj
*	    if incode then
	    tst.l  	d5
	    beq	dblqinterp
	   	move.l	#$201F,-(a7)	* move.l	(a7)+,d0
	   	move.l	#$205F,-(a7)	* move.l	(a7)+,a0
	   	move.l	#$4A80,-(a7)	* tst.l		d0
	   	move.l	#$6700,-(a7)	* beq		
	   	move.l	#$0004,-(a7)	*		*+4
	   	move.l	#$4ED0,-(a7)	* jmp		(a0)
*	    else
	    bra	Interp
dblqinterp:
*		d0 = tos+
		move.l	(a7)+,d0
*		a0 = tos+
		move.l	(a7)+,a0
*		if d0 == 0 then 
		tst.l	d0
*			goto Interp
			beq  Interp
*		endif		
*		goto 	(a0)
		jmp	(a0)
*	    endif
*	  endif
ctosj:
*	  if incode then
	  tst.l	d5
	  beq	ctosjmpsym
*		Copy code from cjmptos on to stack:
		lea.l	cjmptos,a0
		clr.l	d0
		lea.l	ctosjmpsym,a1
ctosjlp:
		move.w	(a0)+,d0
		move.l	d0,-(a7)
		cmp.l	a1,a0
		blt     ctosjlp
		move.l	d1,-(a7)	* put non-? from peek ahead above back
		bra	WithCh
cjmptos:				* Code copied to code function
		tst.l	(a7)+		* test and pop tos
		beq	ctosjmpsym	* label is the right distance away at run time
		lea.l	LNGBYTS(a7),a7	* drop label if false
		move.l	(a7)+,d0	* pop i
		halfmark		* don't clobber a6 yet
	  	  move.l d0,-(a7)	* push i this side of mark
	  	  move.l (a6),-(a7)	* push v
	  	  lea.l	 MARKBYTS+LNGBYTS(a7),a6	* set FrmPtr=a6 properly
	          clr.l  -(a7)		* push value=false so get addr
	          lea.l	 VecEltI,a0	
	          jsr	(a0)
	        unmark			* leave addr of code byte in tos
	        move.l	(a7)+,a0	* pop it into a0
	        jmp	(a0)		* and away we go!
*	  else
ctosjmpsym:
	  	tst.l	(a7)+
	  	bne	tosjmpsym
	  	lea.l	LNGBYTS(a7),a7	* drop label if false
* 	  endif
	move.l	d1,-(a7)		* put non-? from peek ahead above back
	bra	WithCh
*
*	======================= . Include File ========================================
*				* User should really surround the operator with mark,unmark;
*				* the file is treated like a symbolic function whose body is
*				* only visible FilBuf chars at a time. The interpretor will
*				* NOT backup or skip forward out out the FilBuf chars it is
*				* working on at a time.
tryperiod:
	cmpi.l	#Char_DOT,d0	* . is include of file named in tos chvec at interp time
	bne	trycomma	* note the chvec is usually quoted to protect dot inside it!	
 	  mark
		lea.l	FilClose,a0
		jsr	(a0)
	  unmarkdrop			* discard returned tos
	  mark
	    	pea.l	_Fid		* ptr to fid name
	    	move.l	a3,-(a7)	* start at DicLast
	    	lea.l	DicSrch,a0
	    	jsr	(a0)
	  unmark			* ptr to _fid dicent in tos
	  move.l	(a7)+,d1	  
	  move.l	(a7)+,d0	* chvec ptr to filename
	  mark
	  	move.l	d1,-(a7)	* ptr to fid Dic
		move.l	d0,-(a7)	* chvec ptr to filename
		lea.l	FilOpen,a0
		jsr	(a0)
	  unmark			* tos FilOpen Result, fid updated
	  move.l	(a7),d0
*	  cmpi.l	#FILOK,d0	* since FILOK = 0 this line commented out
	  bne		Interp		* leave it on stack if error
	  lea.l	        LNGBYTS(a7),a7	* drop it if OK
	  move.l	d7,a0
  	  move.l  	d6,-(a7)        	* ret StrPrt
	  move.l	-_GL_FilBuf(a0),d6 * New SymStr, ISFN=FALSE
* 	  Caller must mark (and maybe push any args) before the include.
	  move.l	d6,(a6)			* enables ^ to work in include files
          move.l  	a5,-(a7)        	* ret SymPC
          move.l  	#FILBUFMAXIX,d0		* new..
          move.l  	d0,a5			* ..SymPC
	bra     	Interp		* into the included file's first buffer full
*	return from this point is located at in StrNxtCh when hit end of Fil; unmark is callers
*
*	=======================  , Display Stack TOS .. TOS-7 ========================================
trycomma:
	cmpi.l	#Char_COMMA,d0		* , display space, the 8 hex dig of tos, tos-1, tos-2, tos-3
	bne	tryuparrow		* leaves all in place at interp time
*		a1 = 0
		clr.l		d0
		move.l		d0,a1
*		drop PrtDispChLong(NUL,@LNGBYTS+MARKBYTS(a7,a1.l),LNGNIBS-1)
		halfmark
			clr.l	-(a7)
			lea.l	LNGBYTS+MARKBYTS(a7,a1.l),a0
			lea.l	MARKBYTS(a7),a6	* new frame
			move.l	a0,-(a7)
			move.l	#LNGNIBS-1,-(a7)
			lea.l	PrtDispChLong,a0
			jsr	(a0)
		unmarkdrop
*		drop PrtDispKey(COLON)
		mark
			move.l	#Char_COLON,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
		moveq.l		#BUFCHLS-1,d4	* display 8 longs in a line
*		repeat
commaloop:	
*			drop PrtDispChLong(SPACE,@(a7,a1.l),LNGNIBS)
			move.l	(a7,a1.l),d0	* nxt long to display
			mark
				move.l	#Char_SPACE,-(a7)
				move.l	d0,-(a7)
				move.l	#LNGNIBS,-(a7)
				lea.l	PrtDispChLong,a0
				jsr	(a0)
			unmarkdrop
*			a1 += LNGBYTS
			add.l	#LNGBYTS,a1
* 		until (d4-- == -1)
		dbf	d4,commaloop
*
*		drop  PrtDspKey(CR)
		mark
			move.l	#Char_CR,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
*		drop PrtDispKey(LF)
		mark
			move.l	#Char_LF,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
*
*		=========================== ,, Display Stack TOS-8 .. TOS-15 ========================
*		tos = StrNxtCh()
*		if tos <> #Char_COMMA then 
		mark
			lea.l	StrNxtCh,a0
			jsr	(a0)
		unmark
		move.l	(a7),d0
		cmpi.l	#Char_COMMA,d0
*			goto withch
			bne	Withch
*		endif
*						* ,, displays tos .. tos-7 crlf then tos-8..tos-15
*		drop tos
		lea.l		LNGBYTS(a7),a7	* drop the 2nd ,
*		a1 = 8*LNGBYTS
		move.l		#BUFCHLS*LNGBYTS,d0
		move.l		d0,a1
*		drop PrtDispChLong(NUL,@LNGBYTS+MARKBYTS(a7,a1.l),LNGNIBS-1)
		halfmark
			clr.l	-(a7)
			lea.l	LNGBYTS+MARKBYTS(a7,a1.l),a0
			lea.l	MARKBYTS(a7),a6	* new frame
			move.l	a0,-(a7)
			move.l	#LNGNIBS-1,-(a7)
			lea.l	PrtDispChLong,a0
			jsr	(a0)
		unmarkdrop
*		drop PrtDispKey(COLON)
		mark
			move.l	#Char_COLON,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
		move.l		#7,d4		* 8 more words to display in next line
*		repeat
commaloop2:
*			drop PrtDispChLong(SPACE,(a7,a1.l),LNGNIBS)
			move.l		(a7,a1.l),d0	* nxt long to display
			mark
				move.l	#Char_SPACE,-(a7)
				move.l	d0,-(a7)
				move.l	#LNGNIBS,-(a7)
				lea.l	PrtDispChLong,a0
				jsr	(a0)
			unmarkdrop
*			a1 += LNGBYTS
			add.l	#LNGBYTS,a1
* 		until (d4-- == -1)
		dbf	d4,commaloop2
*
*		drop PrtDispKey(CR)
		mark
			move.l	#Char_CR,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
*		drop PrtDispKey(LF)
		mark
			move.l	#Char_LF,-(a7)
			lea.l	PrtDispKey,a0
			jsr	(a0)
		unmarkdrop
*
	bra Interp
*
*	=======================  ^ GOTO (TOS) ========================================
*				* This is difficult to use!  Interpretor will only branch within the body of
*				* the function/literal/include file buffer at hand now. It will NOT
*				* backup beyond the last read or go forward to the next one.
*				* This works in code functions too.  The index value in TOS
*				* will frequently have to be adjusted prior to use with this
*				* operator and there must be a suitable landing area identified.
*				* Jumping the interpretor into a comment may be disasterous. 
*				* Jumping in a code function beyond the last instruction may be unfortunate.
*				* If you want to return early from a function/literal/include file use a
*				* return operator staring with " e.g. " "d "? "~ 
tryuparrow:
	cmpi.l	#Char_UPARROW,d0 * jmp to code at addr (tos)if double ^^  or v[(tos)] if single ^
	bne	startname	* Names, incl predefs, cannot start with any of the single chars tested above.
	mark			* peek ahead to see if ^^ or just ^
		lea.l	StrNxtCh,a0
		jsr	(a0)	
	unmark	
	move.l	(a7)+,d1	
	cmpi.l	#Char_UPARROW,d1
	bne	tosjmp
	  tst.l d5
	  beq   dbluparrowint
	  	move.l	#$205F,-(a7)	* move  (a7)+,a0
	  	move.l	#$4ED0,-(a7)	* jmp	(a0)
	  bra	Interp
dbluparrowint:	        
*				* ^^ jmps through (tos) addr not ix
	  move.l (a7)+,a0
	  jmp	 (a0)		* adios...
*
*				* does the jump at interp time or assembles code to stk for runtime jump
tosjmp:				* jmp to addr in v[ix=tos] using vecptr from result slot at call
	tst.l	d5
	beq	tosjmpsym
incodetosjmp:
*		Copy code from jmptos on to stack:
		lea.l	jmptos,a0
		clr.l	d0
		lea.l	tosjmpsym,a1
tosjlp:
		move.w	(a0)+,d0
		move.l	d0,-(a7)
		cmp.l	a1,a0
		blt     tosjlp
		move.l	d1,-(a7)	* put non-^ from peekahead above back
		bra	WithCh
jmptos:					* Code copied to code function
		move.l	(a7)+,d0	* pop i
		halfmark		* don't clobber a6 yet
	  	  move.l d0,-(a7)	* push i this side of mark
	  	  move.l (a6),-(a7)	* push v
	  	  lea.l	 MARKBYTS+LNGBYTS(a7),a6	* set FrmPtr=a6 properly
	          clr.l  -(a7)		* push value=false so get addr
	          lea.l	 VecEltI,a0	
	          jsr	(a0)
	        unmark			* leave addr of code byte in tos
	        move.l	(a7)+,a0	* pop it into a0
	        jmp	(a0)		* and away we go!
*
tosjmpsym:				* don't worry about losing peekahead char; this is an interp jmp
	move.l	(a7)+,a5		* Much easier for symcode fn, just pop off SymPC
	bra	Interp			* and go get next byte (target known in same fn)
*
*	=======================  @ Namecall ========================================

namecall:	* build anonymous dic entry out in flat for vector whose pointer is in tos
	move.l	(a7)+,d2	* vecptr
	cmp.l	a2,d2
	beq	ncok		* @ following last vecptr on stk requests building anon dicent
	cmp.l	a3,d2
	bne	ncscalar
	move.l	d2,-(a7)	* put it right back
	bra	Interp		* @ following last dicptr on stk is no-op
*
ncscalar:	*	scalar name call requests building anon scalar dicent w/ tos value 
	mark
	  move.l	a3,-(a7)	* prev
	  clr.l	-(a7)		* null nm ptr
	  moveq.l	#DicTYPSCALAR,d0
	  move.l	d0,-(a7)	* scalar
	  move.l	d2,-(a7)
	  lea.l	DicNew,a0
	  jsr	(a0)
	unmark
	bra	Interp
ncok:
	mark
	  move.l d2,-(a7)
	  lea.l	 VecElSz,a0
	  jsr	 (a0)
	unmark
	move.l	 (a7)+,d1
	cmpi.l	 #Vec_ELSZ_BYT,d1
	bne	 nc1
	move.l	 #DicTYPSYMFN,d1
	bra	 nc3
nc1:
	cmpi.l	 #Vec_ELSZ_WRD,d1
	bne	 nc2
	move.l	 #DicTYPBINFN,d1
	bra	 nc3
nc2:
*	\\	#Vec_ELSZ_LNG,Vec_ELSZ_DBL
	move.l	 #DicTYPVEC,d1
nc3:
	mark
	  move.l a3,-(a7)	* prev
	  clr.l	 -(a7)		* null is anonymous name ptr
	  move.l d1,-(a7)	* DicType per vec size decode above
	  move.l d2,-(a7)	* vecptr
	  lea.l	 DicNew,a0
	  jsr	 (a0)		* tos is DicPtr to anonymous DicEnt with correct type and vecptr
	unmark
	bra	Interp
*
*	=======================  NAME ========================================
startname:	* Start of defined or undefined name (don't get here for chars handled above)
*       Scan name to stack until get to any of:
*           	<any ctl char including space> DEL
*		' # $ @ \
*		` % ^ { } : ; " , ? 
*	but following chars will just be added to the name:
*		~ ! & | * + - / = < > ( ) [ ] _ .
*	Scan ending ch is in tos
_sn_vdpr	equ	-LNGBYTS
_sn_Result	equ	0
_sn_Prv		equ	LNGBYTS
*
  	move.l	d7,a0
	move.l	-_GL_NmBuf(a0),d1
	move.l	d1,-(a7)    * vdpr VecPtr(if chlit w/o qtes),DicPtr(if nmcall),or result(if invoke)
	mark		 		* Result will be ch ending the scan
	  move.l	d0,-_sn_Result(a6) * Result = 1st char of name from entry
*	  vdpr = VecSetCurIx(-_GL_NmBuf(d7),-1)
	  mark
		move.l	d7,a0
		move.l	d1,-(a7)
		move.l	#VECEMPTYIX,-(a7)
		lea.l	VecSetCurIx,a0
		jsr	(a0)
	  unmark
	  move.l	(a7)+,-_sn_vdpr(a6)
*	  repeat 
nxtnmch:
*	    vdpr = VecAppend(nmv,0,ch)
	    halfmark
		move.l	-_sn_vdpr(a6),-(a7)
		clr.l	-(a7)
		move.l	-_sn_Result(a6),-(a7)
		lea.l	MARKBYTS+2*LNGBYTS(a7),a6
		lea.l	VecAppend,a0
		jsr	(a0)
	    unmark
	    move.l	(a7)+,-_sn_vdpr(a6)
*	    Result = StrNxtCh()
	    mark		* each time around we call StrNxtCh and push one more ch
			lea.l	StrNxtCh,a0
			jsr	(a0)
	    unmark
	    move.l	(a7)+,d0	* ch
	    move.l	d0,-_sn_Result(a6)	* ch ending scan if exit taken
*	  until ch in {' ','#','$','''','@','\','%','^','{','}',':','"',';',',','`','?',$7f}
	  cmpi.l	#Char_SPACE,d0
	  ble	Nmlookup
	  cmpi.l	#Char_POUND,d0
	  beq	Nmlookup
	  cmpi.l	#Char_DOLLAR,d0
	  beq	Nmlookup
	  cmpi.l	#Char_LQUOTE,d0
	  beq	Nmlookup
	  cmpi.l	#Char_AT,d0
	  beq 	Nmlookup
	  cmpi.l	#Char_BKSLASH,d0
	  beq	Nmlookup
*
	  cmpi.l	#Char_PCT,d0
	  beq	Nmlookup
	  cmpi.l	#Char_UPARROW,d0
	  beq	Nmlookup
	  cmpi.l	#Char_LBRACE,d0
	  beq	Nmlookup
	  cmpi.l	#Char_RBRACE,d0
	  beq	Nmlookup
	  cmpi.l	#Char_COLON,d0
	  beq	Nmlookup
	  cmpi.l	#Char_DBLQUOTE,d0
	  beq	Nmlookup
	  cmpi.l	#Char_SEMI,d0
	  beq	Nmlookup
	  cmpi.l	#Char_COMMA,d0
	  beq	Nmlookup
	  cmpi.l	#Char_RQUOTE,d0
	  beq	Nmlookup
	  cmpi.l	#Char_QMARK,d0
	  beq	Nmlookup
	  cmpi.l	#Char_DEL,d0
 	  beq	Nmlookup
	  bra	nxtnmch
Nmlookup:
*	  vdpr = VecAppend(-_GL_NmBuf(d7),0,0) * append gratuitious null
	  halfmark
		move.l	-_sn_vdpr(a6),-(a7)
		lea.l	MARKBYTS(a7),a6
		clr.l	-(a7)
		clr.l	-(a7)
		lea.l	VecAppend,a0
		jsr	(a0)
	  unmark	
	  move.l	(a7)+,d0
	  move.l	d0,-_sn_vdpr(a6)	
	unmark			* tos is ch ending scan, tos-1 is vdpr
*
*	==== Names are CASE SENSITIVE; UNDEFINED NAMES Autodefine Character Literals; TYPE CAREFULLY! 
*       if dicsrch(vdpr)== null then 
	mark
		move.l	d0,-(a7) * chvecpt from VecNew
		move.l	a3,-(a7) * start at DicLast
		lea.l	DicSrch,a0
		jsr	(a0)
	unmark				* dicptr, ch, vecptr 
	move.l	(a7)+,d0	* Null ptr = undefined, tos is ch,tos-1 is vdpr
*		goto Withch
		beq	WithCh		* ch ending scan is in tos, vdpr is in tos-1
*	endif
	move.l	d0,LNGBYTS(a7)	* defined: therefore copy dicptr to vdpr
*
defdname:
* 	tos is ch that ended name, tos-1 is DicPtr
*       Arrive here by fall in or jmp NOT call! Tos is NOT ret pc-
	move.l	(a7),d0
	cmpi.l	#Char_AT,d0
	bne	invoke
*	Dropping the @ is all we need to do for defined name namecalls
	lea.l	LNGBYTS(a7),a7	* discard the at-sign
	bra	Interp
	
*
*	=======================  INVOKE ========================================
invoke:
*       Here with tos-1 containing valid DicPtr which was followed by 
*	ch in tos which was not @ immediately after name
	OFFSET	0
ivkdic	DS.L	1
ivkch	DS.L	1
ivkres	DS.L	1
ivkprvf	DS.L	1
ivkdic2 DS.L	1
	ORG	*
	mark
		move.l	MARKBYTS+LNGBYTS(a7),-(a7) * Push DicPtr
		lea.l	DicType,a0
		jsr	(a0)
	unmark				* now have DicTyp,ch,DicPtr
	move.l	(a7)+,d0		* now have ch,DicPtr
	cmpi.l	#DicTYPSCALAR,d0
	beq	InvokeScalVec
	cmpi.l	#DicTYPVEC,d0
	bne	invktryfns
*
InvokeScalVec:
	mark
		move.l	MARKBYTS+LNGBYTS(a7),-(a7) * DicPtr,ch,DicPtr
		lea.l	DicScalVecP,a0	* doesn't alter d0
		jsr	(a0)
	unmark				* VecPtr,ch,Dicptr
	move.l	(a7)+,d1		* ch,Dicptr
	move.l	d1,LNGBYTS(a7)		* ch,Vecptr
	bra	WithCh
*
invktryfns:				* ch,DicPtr
	mark
		move.l	MARKBYTS+LNGBYTS(a7),-(a7) * DicPtr,prev frm, reslt,ch,DicPtr
		lea.l	DicScalVecP,a0	* doesn't alter d0
		jsr	(a0)
	unmark				* FnVecPtr, ch, DicPtr
	cmpi.l	#DicTYPBINFN,d0
	bne	InvokeSym
*
InvokeBin:
*       Invoke Binary Code Function Here
	move.l	(a7),d0			* v,ch, DicPtr
	mark
		move.l	d0,-(a7)	* v
		clr.l	-(a7)		* i=0
		clr.l	-(a7)		* val=false
		lea.l	VecEltI,a0
		jsr	(a0)
	unmark				* @v[0],v,ch,dicptr
	move.l	(a7)+,a0		* v,ch,DicPtr
	move.l  (a7)+,(a6)		* ch,DicPtr; res at call is v so ` (goto) works
	lea.l	MARKBYTS(a7),a7		* old stack
*	Function user had to push mark and args, result=v
        jsr     (a0)		        * and away... we go
* 	Function user has to unmark too
        bra     Interp       
*
InvokeSym:				* SymFn Vecptr, ch,DicPtr
	move.l	(a7)+,a0		* For new SymStr: ch,DicPtr
	lea.l	2*LNGBYTS(a7),a7	* drop ch, dicptr
*	Function user had to push mark and args, SymStr and SymPC are last 2 args
InvokeSym1:
	move.l  d6,-(a7)        	* ret StrPrt
        move.l  a5,-(a7)        	* ret SymPC
        moveq.l	#VECEMPTYIX,d0		* new..
        move.l  d0,a5			* ..SymPC
	move.l  a0,d6			* new SymStr
        ori.l   #StrBufISFN,d6		* ISFN=TRUE
        move.l	d6,(a6)			* result at call is set to SymStr
	bra     Interp			* can dump ending white,ctl,del
*	return from this point is located at in StrNxtCh when hit end of Str
*	caller must unmark
*
*	=======================  ' CHAR LITERAL ========================================
startchlit:     * Start of Character Literal (byte element size)
*       RQuote starts and Rquote or unescaped CR ends such a literal
*	Char literals ALWAYS contain a final element which a null (zero) byte.
*       Can use escape \ inside to let next character be
*       included (even white space, ctl chars, \ itself, ', #, or$
	mark
	  lea.l		-MARKBYTS(a7),a0 * calculate datap
	  move.l	a0,-(a7)	* save datap at -8(a6)
*	  cycle				* here still scanning chars into the literal
chnxt:					* need to go get another one
*		tos = StrNxtCh() 
	 	mark
	    		lea.l	StrNxtCh,a0	
	    		jsr	(a0)
	  	unmark
*	  	cycle			* here scanning with a ch already which may end char lit
chnxttest:
*			d0 = tos
	  		move.l	(a7),d0		* ch
*			if (d0 == RQUOTE) or (d0 == CR) or (d0 == NUL) then goto chend
	 		  cmpi.l	#Char_RQUOTE,d0	
	 		  beq		chend
	  		  cmpi.l	#Char_CR,d0
	  		  beq		chend		* unescaped EOL is implicit trailing RQUOTE
	  		  cmpi.l	#Char_NUL,d0
	  		  beq		chend		* EOL sometimes turned into NUL
*	  		endif
*			if (d0 <> BKSLASH) then
	  		cmpi.l		#Char_BKSLASH,d0
*				goto 	chnxt		* all multi-char sequences start w/ BKSLASH
	  			bne	chnxt		* unless BKSLASH just take the character
*	  		endif
*
*	  						* Char Lit \ code, enter with tos=d0=1st \
*			drop tos
	  		lea.l		LNGBYTS(a7),a7	* discard the opening backslash
*			d0= tos = StrNxtCh()
	  		mark
	  			lea.l	StrNxtCh,a0
	  			jsr	(a0)
	  		unmark				* alloc tos = ch immed after 1st \
	  		move.l	(a7),d0			* character immed after \ in  tos=d0
*							* simple two-char \-sequences first
*			if (d0 == BKSLASH) or (d0==RQUOTE) or (d0==CR) or (d0==NUL) then goto chnxt
	  		  cmpi.l	#Char_BKSLASH,d0
	 		  beq		chnxt   * \\ becomes \
	  		  cmpi.l	#Char_RQUOTE,d0
	  		  beq		chnxt	* \' becomes '
			  cmpi.l	#Char_CR,d0
			  beq		chnxt	* \ cr becomes cr
			  cmpi.l	#Char_NUL,d0
			  beq		chnxt	* \ nul becomes nul
*			endif
*						* last try simple \ch is when ch not hexdigit
*			d2 = tos+		* get hexval of the ch and bail if not hex
	  		move.l	(a7)+,d2	* save copy of ch in d2
*			if (d0 = CharHexVal(d2)) < 0 then goto chnxthexend
	  		  mark		
	  			move.l	d2,-(a7)
	  			lea.l	CharHexVal,a0
	  			jsr	(a0)
	  		  unmark	  
	  		  move.l	(a7)+,d0	* d0 =-1 if nonhex, else val
	 		  blt		chnxthexend
*			endif
*						* chnxthexlop code covers \
*						* now loop while in hex digits
			move.l	d0,-(a7)	* alloc accum val w/ 1st nib in it
*			cycle {			* \ followed by n>=1 hexdigits accum'd in tos
chnxthexlp:
*				d2 = StrNxtCh()	* get next character
				mark
					lea.l	StrNxtCh,a0
					jsr	(a0)
				unmark
				move.l	(a7)+,d2	* d2 is the char
*
*				d0 = CharHexVal(d2)	* non hex will yield -1
				mark
					move.l	d2,-(a7)
					lea.l	CharHexVal,a0
					jsr	(a0)
				unmark
				move.l	(a7)+,d0	* d0 is the hexval
*				if d0 < 0 then	
*					goto chnxthexend	
					blt	chnxthexend
*				endif
*				* it is valid hexval
*				tos = (tos << 4) || (d0 && $F)
				move.l	(a7)+,d1
				lsl.l	#NIBBITS,d1		* let byte overflow to left
				and.l	#NIBMSK,d0
				or.l	d0,d1
				move.l	d1,-(a7)
*			end cycle		* \ followed by n+1>=1 hex digits accum'd in tos	
			bra	chnxthexlp
chnxthexend:			* have char in d2 which is not hex, 
*				* if \ had hexdigits they are accumulated in tos-1 
*				* push d2 into tos, whatever we accumul'd if any is in tos-1
				move.l	d2,-(a7)
*		end cycle	* already have next character
		bra	chnxttest	
*	  end cycle
*
chend:				* here with character which ended char lit
	  move.l	a7,d1
	  move.l	-MARKBYTS(a6),d0
	  clr.l	(a7)		* Replace final delimiter by magic single null
				* String contains at least one ch, last one is null
	  subq.l	#LNGBYTS,d1		* Include trailing null
	  sub.l	d1,d0		* number unpacked chars on long stack
	  asr.l	#2,d0		* number of chars
	  move.l	d0,d1
	  addq.l	#7,d1
	  asr.l	#3,d1		* round up to integral number of doubles
	  asl.l	#2,d1		* back to number of wds
	  addq.l	#VEC_HDR_WDS,d1		* Add header word count	
	  move.l	-MARKBYTS(a6),-(a7)	* datap
	  move.l	a2,-(a7)	* prev
	  move.l	d1,-(a7)	* VecDcWdSz
	  moveq.l	#Vec_ELSZ_BYT,d1
	  move.l	d1,-(a7) 	* Vec_ELSZ_BYT
	  subq.l	#1,d0		* VecCrIx
	  move.l	d0,-(a7)
	  lea.l	VecNew,a0
	  jsr	(a0)
	unmark
	bra	Interp		* need another character
*
*	=======================  $ CODE FUNCTION LITERAL ========================================
starthxlit:     * Start of binary code function vector
*       (word element size) Used as assembler for binary code fns.
*	Code function always contains a final element which is a 68000 RTS instruction.
*
	OFFSET	0
_HL_Result	DS.L	1	* result
_HL_OldFrm	DS.L	1	* Prev Frame Ptr
_HL_datap	DS.L	1	* datap
	ORG	*
*
	mark
	  lea.l		-LNGBYTS(a7),a7		* allocate datap
	  lea.l		-LNGBYTS(a7),a0		* datap will store addr of first data word
	  move.l	a0,-_HL_datap(a6)	* set datap
	  move.l	a6,d5			* Incode -> my result
	  bra	Interp
*
*	Explanation of how binfn label and goto works:
*
*	Defining a label:
*	{ {_VecLast `} 'lbl1' #0 : _DicNew} `
*				 \ defines scalar lbl1 with val of cur ix in bincodvec
*	...			\ ...intervening code...
*	lbl1 ^			\ invokes label1 which pushes the ix value from the scalar
*				\ and then ` operator computes addr of ixth word in code
*				\ using VecEltI (code vecptr is put in result slot of
*				\ code fn when it is invoked so it can be used to 
*				\ replace ix by address of the actual code byte,
*				\ then it is jumped to directly
*
*				\ For fwd ref lbls must define label1 first, then
*				\ use _DicSetScalVecP to update it when get to correct : locn
*				\ else lbl1 becomes anonymous chlit when 1st encountered!
*
midhxlit:				* There is an open mark for the @bincodfn result
	  move.l	a7,d2		* d2-> lowest addr data on long stk
	  move.l	-_HL_datap(a6),d1 * d1-> highest addr data on long stk
	  addq.l	#LNGBYTS,d1	* include first long itself
	  sub.l		d2,d1		* d1=#bytes data on long stk
	  bgt		endmidhx1
	  clr.l		-_HL_Result(a6) * Null Ptr since no code defined
	  bra		endmidhex3
endmidhx1:
	  asr.l		#2,d1		* d1 is # of words on long stack
	  move.l	d1,-_HL_Result(a6) * store the index in my result
endmidhex3:
	  unmark
	  bra		WithCh		* already had next char sorting out possible : options
*
endhxlit:				* There is an open mark for the @bincodfn result
	  move.l	#$4E75,-(a7)	* Final rts to minimize cd fn runoff; $$ assembles just an rts
	  move.l	a7,d2		* d2 -> lowest addr data on long stk
	  move.l	-_HL_datap(a6),d1 * d1 -> highest addr data on long stk
	  move.l	d1,-(a7)	* datap
	  move.l	a2,-(a7)	* prev = VecLst
	  addq.l	#LNGBYTS,d1	* include the first long itself
	  sub.l		d2,d1		* d1 = # bytes data on long stk
	  asr.l		#2,d1		* d1 is # of words on long stack	
	  move.l	d1,d2
	  addq.l	#3,d2
	  andi.l	#VecDSMsk,d2	* d2 = rounded up to integral multiple of doubles
	  addq.l	#VEC_HDR_WDS,d2		* d2 = VecDcWdSz including 4 word header
	  move.l	d2,-(a7)	* VecDWSz
	  moveq.l 	#Vec_ELSZ_WRD,d2
	  move.l	d2,-(a7)	* VecElSz
	  subq.l	#1,d1		* convert to 0-rel inx of last used
	  move.l	d1,-(a7)	* VecCrIx
	  lea.l		VecNew,a0
	  jsr		(a0)
endhex2:
	  clr.l		d5		* incode=FALSE
endhex3:
	unmark
	bra		Interp		* Need another character
*	endif
*
*	=======================  # HEX SCALAR ========================================
*	# followed by non-hex-digit is same as #0.
startscalar:    * Start hexadecimal coded long literal
	mark			* Simulate entry by call
*	  cycle
scalarnxt:
*		d0 = StrNxtCh(SymPrt,SymPC)
		mark
		  lea.l	StrNxtCh,a0
		  jsr 	(a0)
		unmark
		move.l	(a7)+,d0
*		d2 = d0		* save ch in case exit to scalend
		move.l	d0,d2
*		d0 = CharHexVal(d0)
		mark
		  move.l d0,-(a7)
		  lea.l	 CharHexVal,a0
		  jsr	 (a0)
		unmark
		move.l	(a7)+,d0
*		if d0 >= 0 then	* valid hexdigit
		blt	scalend
*			result = (result << 4) | d2
			move.l	(a6),d1
			asl.l	#NIBBITS,d1
			or.l	d0,d1
			move.l  d1,(a6)
*	  	endif
*	  endcycle
	  bra	scalarnxt
scalend:
	unmark
	move.l	d2,-(a7)
	bra 	WithCh		* value is in tos-1 and next sym char in tos
*
*	=======================  0-9 DECIMAL SCALAR ========================================
*	Start decimal literal with optional trailing - to make it negative
declit:
	mark			* Simulate entry by call
*	  cycle
decscalarnxt:
*		d2 = d0
		move.l	d0,d2
*		d0 = CharDecVal(d2)
		mark
			move.l	d2,-(a7)
			lea.l	CharDecVal,a0
			jsr	(a0)
		unmark
		move.l	(a7)+,d0
		blt	decscalend
*		result = (result *10) + (d0)
		move.l	(a6),d1
		move.l	d1,d2
		add.l	d2,d2
		asl.l	#3,d1
		add.l	d2,d1
		add.l	d0,d1
		move.l  d1,(a6)
*		ch = StrNxtCh(SymPrt,SymPC)
		mark
		  lea.l	StrNxtCh,a0
		  jsr 	(a0)
		unmark
		move.l	(a7)+,d0
*	  end
	  bra	decscalarnxt
decscalend:
	unmark
	cmpi.l	#Char_MINUS,d2
	bne	decplus
		neg.l	(a7)
		bra	Interp	* go get another symbol
decplus:
	move.l	d2,-(a7)	* next symbol started immed following declit
	bra 	WithCh		* value is in tos-1 and next sym char in tos

*
*	ORG	(*+3)&-4
EndCode:
	END	START		; last line of source












































































































































































































































































*~Font name~Courier New~
*~Font size~10~
*~Tab type~1~
*~Tab size~8~
